// Khai báo các biến và hằng số
const API_BASE = 'https://xneva.vn/ai_client/webhook/facebook';
const API_TRAINING = 'https://xneva.vn/ai_client/add_training_data';
const localStorageKey = 'aiStatusCache';

// Biến để bật/tắt log - Đặt false để tắt tất cả log debug
const DEBUG_LOGS = false;

// Hàm log có điều kiện
function debugLog(...args) {
    if (DEBUG_LOGS) {
        console.log('[Extension]', ...args);
    }
}

debugLog('Chạy init');

// Hàm kiểm tra trang hiện tại là Pancake hay Facebook
function isPancakePage() {
    return window.location.hostname.includes('pancake.vn');
}

// Hàm trích xuất page_id và pid từ avatar link của Pancake
function extractPancakeIds() {
    // Phương pháp 1: Tìm avatar trong selected-customer
    const selectedCustomer = document.querySelector('.selected-customer');
    if (selectedCustomer) {
        const avatarImg = selectedCustomer.querySelector('img[class*="customer-avatar-"]');
        if (avatarImg) {
            const src = avatarImg.getAttribute('src');
            const className = avatarImg.className;
            
            if (src) {
                // Trích xuất page_id từ URL avatar
                const pageIdMatch = src.match(/\/pages\/(\d+)\/avatar/);
                if (pageIdMatch) {
                    const page_id = pageIdMatch[1];
                    
                    // Trích xuất pid từ URL hoặc class name
                    const pidMatch = src.match(/\/pages\/\d+\/avatar\/(\d+)/);
                    const customerAvatarMatch = className && className.match(/customer-avatar-(\d+)/);
                    
                    const pid = pidMatch ? pidMatch[1] : 
                               customerAvatarMatch ? customerAvatarMatch[1] : 
                               null;
                    
                    debugLog('Phương pháp 1 - Selected customer avatar:', { page_id, pid, src, className });
                    return { page_id, pid };
                }
            }
        }
    }
    
    // Phương pháp 2: Tìm avatar trong chat menu hoặc ở navbar (fallback)
    const avatarImgs = document.querySelectorAll('img[src*="/api/v1/pages/"][src*="/avatar"]');
    if (avatarImgs && avatarImgs.length > 0) {
        for (const img of avatarImgs) {
            const src = img.getAttribute('src');
            if (!src) continue;
            
            // Trích xuất page_id từ URL avatar
            const pageIdMatch = src.match(/\/pages\/(\d+)\/avatar/);
            if (!pageIdMatch) continue;
            
            const page_id = pageIdMatch[1];
            
            // Trích xuất pid từ class name hoặc từ URL
            const pidMatch = src.match(/\/pages\/\d+\/avatar\/(\d+)/);
            const customerAvatarMatch = img.className && img.className.match(/customer-avatar-(\d+)/);
            
            const pid = pidMatch ? pidMatch[1] : 
                       customerAvatarMatch ? customerAvatarMatch[1] : 
                       null;
            
            if (page_id) {
                debugLog('Phương pháp 2 - Avatar img:', { page_id, pid, src });
                return { page_id, pid };
            }
        }
    }
    
    // Phương pháp 3: Tìm từ URL hoặc data attributes
    const chatMenu = document.querySelector('.chat-menu');
    if (chatMenu) {
        // Tìm elements có data attributes
        const elementsWithData = chatMenu.querySelectorAll('[data-page-id], [data-customer-id], [data-pid]');
        for (const element of elementsWithData) {
            const page_id = element.getAttribute('data-page-id');
            const pid = element.getAttribute('data-customer-id') || element.getAttribute('data-pid');
            
            if (page_id) {
                debugLog('Phương pháp 3 - Data attributes:', { page_id, pid });
                return { page_id, pid };
            }
        }
    }
    
    // Phương pháp 4: Tìm từ URL của trang
    const urlPageId = extractPageIdFromUrl();
    if (urlPageId) {
        // Thử tìm pid từ URL hash hoặc query params
        const urlParams = new URLSearchParams(window.location.search);
        const hashMatch = window.location.hash.match(/customer[_-]?(\d+)/i);
        const urlPid = urlParams.get('customer_id') || urlParams.get('pid') || 
                      (hashMatch ? hashMatch[1] : null);
        
        debugLog('Phương pháp 4 - URL:', { page_id: urlPageId, pid: urlPid });
        return { page_id: urlPageId, pid: urlPid };
    }
    
    debugLog('Không thể trích xuất IDs từ tất cả phương pháp');
    return null;
}

// Hàm hiển thị thông báo
function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.style.position = 'fixed';
    notification.style.top = '20px';
    notification.style.right = '20px';
    notification.style.padding = '10px 20px';
    notification.style.borderRadius = '4px';
    notification.style.zIndex = '9999';
    notification.style.maxWidth = '300px';
    notification.style.wordWrap = 'break-word';
    notification.style.transition = 'opacity 0.5s';
    
    if (type === 'success') {
        notification.style.backgroundColor = '#4caf50';
        notification.style.color = 'white';
    } else {
        notification.style.backgroundColor = '#f44336';
        notification.style.color = 'white';
    }

    notification.textContent = message;
    document.body.appendChild(notification);

    setTimeout(() => {
        notification.style.opacity = '0';
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 500);
    }, 3000);
}

// API Training qua background script
async function addTrainingData(currentText, nextText, page_id) {
    try {
        const response = await new Promise((resolve, reject) => {
            chrome.runtime.sendMessage(
                {
                    type: 'api_call',
                    url: API_TRAINING,
                    data: {
                        facebook_page_id: page_id,
                        question: currentText,
                        answer: nextText
                    }
                },
                response => {
                    if (chrome.runtime.lastError) {
                        reject(chrome.runtime.lastError);
                        return;
                    }
                    if (response.success) {
                        resolve(response.data);
                    } else {
                        reject(new Error(response.error));
                    }
                }
            );
        });
        
        if (response.status === 'success') {
            const message = `✅ Đã thêm dữ liệu training cho chatbot "${response.data.chatbot_name}"`;
            showNotification(message, 'success');
        } else {
            const errorMsg = response.message || 'Không thể thêm dữ liệu training';
            showNotification(errorMsg, 'error');
        }
    } catch (error) {
        console.error('[Extension] Lỗi khi thêm training data:', error);
        showNotification('Đã xảy ra lỗi khi thêm dữ liệu training', 'error');
    }
}

// Khởi tạo xử lý tin nhắn một lần
(function initializeMessageHandler() {
    const ICON_TEXT = ' 📘';
    let lastRun = 0;
    let timeoutId = null;
    let retryCount = 0;
    const MAX_RETRIES = 10;
    const RETRY_DELAY = 1000;

    function isUserMessage(el) {
        let node = el;
        for (let i = 0; i < 10; i++) {
            if (!node) return false;
             const img = node.querySelector?.('img[alt]');
                if (img && parseInt(img.getAttribute('height')) > 30) {
                    return true;
                }
            node = node.parentElement;
        }
        return false;
    }

    function findMessageRegion() {
        return new Promise((resolve) => {
            function check() {
                const region = document.querySelector('[role="region"]');
                if (region) {
                    resolve(region);
                } else if (retryCount < MAX_RETRIES) {
                    retryCount++;
                    setTimeout(check, RETRY_DELAY);
                } else {
                    console.warn("[Extension] Không tìm thấy vùng chứa tin nhắn sau nhiều lần thử.");
                    resolve(null);
                }
            }
            check();
        });
    }

    function addLearnIcons() {
        findMessageRegion().then(region => {
            if (!region) return;

            const spans = region.querySelectorAll('span > span');
            spans.forEach((el, index) => {
                const text = el.innerText?.trim();

                const alreadyHasIcon = Array.from(el.childNodes).some(child => {
                    return child.nodeType === Node.TEXT_NODE
                        ? child.textContent.includes('📘')
                        : child.textContent?.includes('📘');
                });


                if (
                    text &&
                    text.length > 0 &&
                    !el.closest('a') &&
                    !el.closest('img') &&
                    !text.startsWith("Sent by") &&
                    !text.startsWith("Seen by") &&
                    !alreadyHasIcon &&
                    isUserMessage(el)
                ) {
                    const icon = document.createElement('span');
                    icon.textContent = ' 📘';
                    icon.style.cursor = 'pointer';
                    icon.title = 'Learn';

                    icon.addEventListener('click', () => {
                        const currentText = el.innerText.replace('📘', '').trim();

                        // Tìm tin nhắn tiếp theo
                        let nextText = '';
                        for (let i = index + 1; i < spans.length; i++) {
                            const next = spans[i];
                            const nextContent = next.innerText?.trim();
                            if (
                                nextContent &&
                                !next.closest('a') &&
                                !next.closest('img') &&
                                !nextContent.startsWith("Sent by") &&
                                !nextContent.startsWith("Seen by")
                            ) {
                                nextText = nextContent;
                                break;
                            }
                        }

                        // Lấy page_id từ URL
                        const params = getQueryParams(window.location.href);
                        const page_id = params['asset_id'];

                        if (!page_id) {
                            showNotification('Không thể xác định Facebook Page ID', 'error');
                            return;
                        }

                        if (!nextText) {
                            showNotification('Không tìm thấy câu trả lời tiếp theo', 'error');
                            return;
                        }

                        debugLog('Learn clicked');
                        debugLog('Current message:', currentText);
                        debugLog('Next message:', nextText);

                        // Gọi API để thêm training data
                        addTrainingData(currentText, nextText, page_id);
                    });

                    el.appendChild(icon);
                }
            });
        });
    }

    // Giới hạn tần suất xử lý mutation: ít nhất 2 giây mỗi lần
    function scheduleAddIcons() {
        const now = Date.now();
        const timeSinceLastRun = now - lastRun;

        if (timeSinceLastRun >= 2000) {
            lastRun = now;
            addLearnIcons();
        } else {
            clearTimeout(timeoutId);
            timeoutId = setTimeout(() => {
                lastRun = Date.now();
                addLearnIcons();
            }, 2000 - timeSinceLastRun);
        }
    }

    // Đợi DOM load xong và khởi tạo
    function initialize() {
        findMessageRegion().then(region => {
            if (region) {
                addLearnIcons();
                const observer = new MutationObserver(scheduleAddIcons);
                observer.observe(region, { childList: true, subtree: true });
            }
        });
    }

    // Đợi DOM load hoàn tất
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initialize);
    } else {
        initialize();
    }
})();

// Hàm helper để parse query parameters
function getQueryParams(url) {
    const params = {};
    const parser = new URL(url);
    for (const [key, value] of parser.searchParams.entries()) {
        params[key] = value;
    }
    return params;
}

// Hàm helper để gọi API thông qua background script
async function fetchJSON(url, data) {
    debugLog('Gửi request đến:', url);
    debugLog('Payload:', data);

    try {
        const response = await new Promise((resolve, reject) => {
            chrome.runtime.sendMessage(
                {
                    type: 'api_call',
                    url: url,
                    data: data
                },
                response => {
                    if (chrome.runtime.lastError) {
                        reject(chrome.runtime.lastError);
                        return;
                    }
                    if (response.success) {
                        resolve(response.data);
                    } else {
                        reject(new Error(response.error));
                    }
                }
            );
        });

        debugLog('Response:', response);
        return response;
    } catch (error) {
        console.error('[Extension] Lỗi gọi API:', error);
        throw error;
    }
}

// Hàm cập nhật nút AI dựa vào trạng thái
function updateAIButton(status, disabled = false) {
    const aiBtn = document.getElementById('ai-status-btn');
    if (!aiBtn) return;

    aiBtn.disabled = disabled;
    if (disabled) {
        aiBtn.innerText = '⚠️ Chưa có trên hệ thống AI';
        aiBtn.style.background = '#fff';
        return;
    }

    if (status == false) {
        aiBtn.innerText = '✨ AI đang chạy';
        aiBtn.style.background = '#fff';
    } else {
        aiBtn.innerText = '⭕ AI đang tắt';
        aiBtn.style.background = '#fff';
    }
}

// Thêm biến kiểm soát để tránh khởi tạo nhiều lần
let isInitializing = false;
let buttonAddingInProgress = false; // Thêm biến này để kiểm soát quá trình thêm nút

// Hàm tạo nút AI
function createAIButton() {
    // Không xóa nút cũ ở đây nữa, sẽ xóa trước khi thêm trong initialize()
    const aiBtn = document.createElement('button');
    aiBtn.id = 'ai-status-btn';
    aiBtn.style.marginRight = '10px';
    aiBtn.style.padding = '6px 8px';
    aiBtn.style.border = '1px solid rgba(0, 0, 0, 0.4)';
    aiBtn.style.borderRadius = '4px';
    aiBtn.style.cursor = 'pointer';
    aiBtn.style.fontSize = '14px';
    aiBtn.style.background = '#fff';
    aiBtn.style.color = '#000';
    aiBtn.innerText = '🔄 Đang kiểm tra trạng thái AI...';
    aiBtn.dataset.timestamp = Date.now(); // Thêm timestamp để phân biệt các nút
    return aiBtn;
}

// Hàm xử lý chuyển đổi trạng thái AI
async function toggleAIStatus(pid, page_id, currentStatus) {
    try {
        const newStatus = !currentStatus;
        const response = await fetchJSON(`${API_BASE}/update_no_reply`, {
            pid,
            page_id,
            is_no_reply: newStatus,
        });
        if (response.status === 'success') {
            updateAIButton(newStatus);
            // Cập nhật localStorage
            const cache = JSON.parse(localStorage.getItem(localStorageKey)) || {};
            cache[pid] = newStatus;
            localStorage.setItem(localStorageKey, JSON.stringify(cache));

            // Gán lại event click với trạng thái mới
            const aiBtn = document.getElementById('ai-status-btn');
            if (aiBtn) {
                // Xóa tất cả event listeners cũ
                const newBtn = aiBtn.cloneNode(true);
                aiBtn.parentNode.replaceChild(newBtn, aiBtn);
                // Thêm event listener mới với trạng thái đã cập nhật
                newBtn.addEventListener('click', () => {
                    toggleAIStatus(pid, page_id, newStatus);
                });
            }
        } else {
            alert('Không thể cập nhật trạng thái AI.');
        }
    } catch (error) {
        console.error('Lỗi khi cập nhật trạng thái AI:', error);
        alert('Đã xảy ra lỗi khi cập nhật trạng thái AI.');
    }
}

// Hàm kiểm tra aiBtn và khởi tạo lại nếu cần
async function checkAndReinitializeAIButton() {
    const aiBtn = document.getElementById('ai-status-btn');
    if (!aiBtn) {
        debugLog('AI Button không tồn tại, khởi tạo lại...');
        await safeInitialize();
    } else {
        debugLog('AI Button vẫn tồn tại');
        
        // Kiểm tra lại trạng thái AI nếu nút đã tồn tại
        if (isPancakePage()) {
            // Xử lý cho Pancake
            const pancakeIds = extractPancakeIds();
            if (pancakeIds) {
                const nameElement = document.querySelector('.copyable-text.customer-name');
                const fullname = nameElement ? nameElement.textContent.trim() : '';
                
                // Kiểm tra lại trạng thái AI cho Pancake
                checkAIStatus(aiBtn, pancakeIds.page_id, pancakeIds.pid, fullname);
            }
        } else {
            // Xử lý cho Facebook
            const url = window.location.href;
            const params = getQueryParams(url);
            const page_id = params['asset_id'];
            const uid = params['selected_item_id'];
            
            if (page_id && uid) {
                // Tìm tên người dùng
                const container = document.querySelector('[data-pagelet="BizInboxDetailViewHeaderSectionWrapper"]');
                if (container) {
                    const nameElement = Array.from(container.querySelectorAll('div'))
                        .find(div => {
                            const text = div.textContent.trim();
                            const style = div.getAttribute('style') || '';
                            return (
                            text &&
                            style.includes('-webkit-line-clamp: 1')
                            );
                        });
                    const fullname = nameElement ? nameElement.textContent.trim() : '';
                    
                    // Kiểm tra lại trạng thái AI
                    checkAIStatus(aiBtn, page_id, uid, fullname);
                }
            }
        }
    }
}

// Hàm an toàn để khởi tạo, tránh khởi tạo đồng thời
async function safeInitialize() {
    if (isInitializing) {
        debugLog('Đang trong quá trình khởi tạo, bỏ qua yêu cầu mới');
        return;
    }
    
    // Xóa tất cả các nút AI hiện có để tránh trùng lặp
    const existingButtons = document.querySelectorAll('#ai-status-btn');
    if (existingButtons.length > 0) {
        debugLog(`Tìm thấy ${existingButtons.length} nút AI hiện có, xóa tất cả`);
        existingButtons.forEach(btn => btn.remove());
    }
    
    isInitializing = true;
    try {
        await initialize();
    } finally {
        isInitializing = false;
    }
}

// Hàm chính để khởi tạo script
async function initialize() {
    let page_id, uid, fullname = '';
    debugLog('Chạy hàm init');

    if (isPancakePage()) {
        // Xử lý cho trang Pancake.vn
        const pancakeIds = extractPancakeIds();
        if (!pancakeIds) {
            debugLog('Không thể xác định page_id và pid từ Pancake');
            
            // Try to at least get page_id from URL for polling
            const urlPageId = extractPageIdFromUrl();
            if (urlPageId) {
                page_id = urlPageId;
                debugLog('Lấy được page_id từ URL:', page_id);
                startPancakeInboxPolling(page_id);
            }
            
            return;
        }
        
        // Lưu lại pancakeIds hiện tại để theo dõi thay đổi
        lastPancakeIds = pancakeIds;
        
        page_id = pancakeIds.page_id;
        uid = pancakeIds.pid; // Ở Pancake, uid chính là pid
        
        // Tìm tên người dùng từ trang Pancake
        const nameElement = document.querySelector('.copyable-text.customer-name');
        fullname = nameElement ? nameElement.textContent.trim() : '';
        
        // Chỉ gọi startPancakeInboxPolling nếu chưa được gọi từ initEarlyPancakePolling
        if (!pancakeCheckInboxInterval) {
            startPancakeInboxPolling(page_id);
        } else {
            debugLog('Pancake inbox polling đã được khởi động trước đó');
        }
    } else {
        // Stop Pancake polling if we're not on Pancake anymore
        stopPancakeInboxPolling();
        stopPancakeConversationChecking();
        // Reset lastPancakeIds khi thoát khỏi Pancake
        lastPancakeIds = null;
        
        // Xử lý cho trang Facebook (giữ nguyên)
        const url = window.location.href;
        const params = getQueryParams(url);
        page_id = params['asset_id'];
        uid = params['selected_item_id'];
        
        if (!page_id) return;
    }

    // Kiểm tra kỹ hơn để tránh trùng lặp
    const existingBtn = document.getElementById('ai-status-btn');
    if (existingBtn) {
        debugLog('AI Button đã tồn tại, kiểm tra lại trạng thái AI cho hội thoại hiện tại');
        
        // Luôn kiểm tra lại trạng thái AI khi có thay đổi hội thoại
        if (page_id && uid) {
            debugLog('Đang kiểm tra trạng thái AI cho hội thoại:', { page_id, uid, fullname });
            checkAIStatus(existingBtn, page_id, uid, fullname);
        }
        
        return;
    }

    let intervalId = null;
    let initAttempts = 0;
    const MAX_INIT_ATTEMPTS = 10;
    
    // Trả về một Promise để có thể đợi quá trình khởi tạo
    return new Promise((resolve) => {
        intervalId = setInterval(() => {
            initAttempts++;
            if (initAttempts > MAX_INIT_ATTEMPTS) {
                debugLog('Vượt quá số lần thử khởi tạo, hủy interval');
                clearInterval(intervalId);
                resolve(false);
                return;
            }
            
            // Nếu đang thêm nút, bỏ qua vòng lặp này
            if (buttonAddingInProgress) {
                return;
            }
            
            let container, actionsArea;
            
            if (isPancakePage()) {
                // Tìm container và actionsArea cho Pancake
                container = document.querySelector('.chat-menu');
                actionsArea = container?.querySelector('.conv-actions');
            } else {
                // Tìm container và actionsArea cho Facebook
                container = document.querySelector('[data-pagelet="BizInboxDetailViewHeaderSectionWrapper"]');
                actionsArea = container?.querySelector('div[role="button"]')?.parentElement?.parentElement;
            }
            
            if (!container || !actionsArea) return;

            // Đánh dấu đang bắt đầu thêm nút
            buttonAddingInProgress = true;
            
            try {
                // Kiểm tra lại xem nút đã tồn tại chưa
                const existingBtn = document.getElementById('ai-status-btn');
                if (existingBtn) {
                    console.log('[Extension] AI Button đã được tạo bởi quá trình khác');
                    clearInterval(intervalId);
                    resolve(true);
                    buttonAddingInProgress = false;
                    return;
                }
                
                // Xóa tất cả các nút có thể tồn tại để đảm bảo không có trùng lặp
                document.querySelectorAll('#ai-status-btn').forEach(btn => {
                    console.log('[Extension] Xóa nút AI tồn tại');
                    btn.remove();
                });
                
                if (!isPancakePage()) {
                    // Tìm tên người dùng trên Facebook
                    const nameElement = Array.from(container.querySelectorAll('div'))
                        .find(div => {
                            const text = div.textContent?.trim();
                            const style = div.getAttribute('style') || '';
                            return (
                            text &&
                            style.includes('-webkit-line-clamp: 1')
                            );
                        });
                    fullname = nameElement ? nameElement.textContent.trim() : '';
                }

                // Tạo nút mới
                const aiBtn = createAIButton();
                
                // Kiểm tra lại một lần nữa trước khi thêm vào DOM
                if (!document.getElementById('ai-status-btn')) {
                    actionsArea.insertBefore(aiBtn, actionsArea.firstChild);
                    console.log('[Extension] Đã thêm nút AI mới:', aiBtn.dataset.timestamp);
                    
                    // Bắt đầu kiểm tra trạng thái AI
                    checkAIStatus(aiBtn, page_id, uid, fullname);
                } else {
                    console.log('[Extension] Không thêm nút vì đã tồn tại');
                }

                clearInterval(intervalId);
                resolve(true);
            } finally {
                // Đánh dấu đã kết thúc thêm nút
                buttonAddingInProgress = false;
            }
        }, 1000);
    });
}

// Hàm kiểm tra trạng thái AI
async function checkAIStatus(aiBtn, page_id, uid, name) {
    debugLog('🔍 BẮT ĐẦU kiểm tra trạng thái AI...', { page_id, uid, name });
    
    // Cập nhật trạng thái nút đang kiểm tra
    aiBtn.innerText = '🔄 Đang kiểm tra trạng thái AI...';
    aiBtn.disabled = true;
    
    try {
        // Đầu tiên, kiểm tra cache để hiển thị trạng thái tạm thời
        const cache = JSON.parse(localStorage.getItem(localStorageKey)) || {};
        
        // Kiểm tra xem có cache cho pid nào liên quan đến uid này không
        const cachedPids = Object.keys(cache);
        for (const cachedPid of cachedPids) {
            const cachedStatus = cache[cachedPid];
            // Hiển thị trạng thái cache tạm thời trong khi đợi API
            if (cachedStatus !== undefined) {
                debugLog('💾 Sử dụng trạng thái cache tạm thời:', cachedStatus);
                updateAIButton(cachedStatus);
                break;
            }
        }

        let pid;
        
        // Nếu đang ở trang Pancake và có pid trực tiếp, không cần gọi check_same_user
        if (isPancakePage() && uid) {
            pid = uid;
            debugLog('🥞 Sử dụng pid trực tiếp từ Pancake:', pid);
        } else {
            // Ở trang Facebook, cần gọi API để lấy pid
            debugLog('📘 Đang gọi API check_same_user cho Facebook...');
            const checkUserResponse = await fetchJSON(`${API_BASE}/check_same_user`, {
                name,
                uid,
                page_id,
            });

            if (checkUserResponse.status !== 'success' || !checkUserResponse.data?.pid) {
                console.error('[Extension] ❌ Không thể xác định user:', checkUserResponse);
                updateAIButton(false, true);
                return;
            }

            pid = checkUserResponse.data.pid;
        }
        
        debugLog('✅ Đã xác định pid:', pid);

        // Kiểm tra cache cho pid cụ thể này
        if (cache[pid] !== undefined) {
            debugLog('💾 Đã tìm thấy trạng thái cache cho pid:', cache[pid]);
            updateAIButton(cache[pid]);
        }

        debugLog('🌐 Đang gọi API check_is_no_reply...');
        const checkReplyResponse = await fetchJSON(`${API_BASE}/check_is_no_reply`, {
            pid,
            page_id,
        });

        if (checkReplyResponse.status === 'success') {
            const is_no_reply = checkReplyResponse.data?.is_no_reply;
            debugLog('✅ Nhận được trạng thái từ API:', is_no_reply);
            updateAIButton(is_no_reply);
            
            // Cập nhật cache
            cache[pid] = is_no_reply;
            localStorage.setItem(localStorageKey, JSON.stringify(cache));

            // Cập nhật event listener
            const currentBtn = document.getElementById('ai-status-btn');
            if (currentBtn) {
                const newBtn = currentBtn.cloneNode(true);
                currentBtn.parentNode.replaceChild(newBtn, currentBtn);
                newBtn.disabled = false;
                newBtn.addEventListener('click', () => {
                    toggleAIStatus(pid, page_id, is_no_reply);
                });
            }
        } else {
            console.error('[Extension] ❌ Lỗi kiểm tra is_no_reply:', checkReplyResponse);
            updateAIButton(false, true);
        }
    } catch (error) {
        console.error('[Extension] 💥 Lỗi khi kiểm tra trạng thái AI:', error);
        updateAIButton(false, true);
    }
    
    debugLog('🏁 KẾT THÚC kiểm tra trạng thái AI');
}

// Khởi tạo hàm kiểm tra nút AI liên tục mỗi giây
function initContinuousButtonCheck() {
    // Hủy interval cũ nếu có
    if (window.aiButtonCheckInterval) {
        clearInterval(window.aiButtonCheckInterval);
    }
    
    // Thiết lập interval mới kiểm tra mỗi 1 giây
    window.aiButtonCheckInterval = setInterval(() => {
        const aiBtn = document.getElementById('ai-status-btn');
        
        if (isPancakePage()) {
            // Kiểm tra đặc biệt cho trang Pancake
            const chatMenu = document.querySelector('.chat-menu');
            const actionsArea = chatMenu?.querySelector('.conv-actions');
            
            // Kiểm tra xem DOM đã sẵn sàng để thêm nút chưa
            if (chatMenu && actionsArea) {
                if (!aiBtn && !isInitializing && !buttonAddingInProgress) {
                    console.log('[Extension] Không tìm thấy nút AI trên Pancake, khởi tạo lại...');
                    
                    // Kiểm tra xem avatar có thay đổi không
                    const pancakeIds = extractPancakeIds();
                    if (pancakeIds && pancakeIds.page_id && pancakeIds.pid) {
                        safeInitialize();
                    } else {
                        console.log('[Extension] Chưa tìm thấy page_id và pid từ Pancake, đợi...');
                    }
                }
            }
        } else {
            // Kiểm tra cho Facebook
            if (!aiBtn && !isInitializing && !buttonAddingInProgress) {
                console.log('[Extension] Không tìm thấy nút AI trên Facebook, khởi tạo lại...');
                safeInitialize();
            }
        }
    }, 1000);
}

// Function to get JWT token from cookies
function getJWTFromCookies() {
    const cookies = document.cookie.split(';');
    let jwtToken = null;
    
    // Look for JWT token in cookies
    for (const cookie of cookies) {
        const [name, value] = cookie.trim().split('=');
        if (name === 'access_token' || name === 'token' || name === 'jwt') {
            jwtToken = decodeURIComponent(value);
            break;
        }
    }
    
    // If not found in common cookie names, try to find any JWT-like token
    if (!jwtToken) {
        for (const cookie of cookies) {
            const value = cookie.split('=')[1];
            if (value && value.includes('.') && value.split('.').length === 3) {
                try {
                    // Try to validate if it looks like a JWT (has 3 parts separated by dots)
                    const parts = value.split('.');
                    if (parts.length === 3) {
                        // Check if middle part can be decoded as base64 (payload)
                        atob(parts[1].replace(/-/g, '+').replace(/_/g, '/'));
                        jwtToken = decodeURIComponent(value);
                        break;
                    }
                } catch (e) {
                    // Not a valid JWT, continue searching
                    continue;
                }
            }
        }
    }
    
    return jwtToken;
}

// Interval ID for pancake inbox checking
let pancakeCheckInboxInterval = null;

// Function to poll the check_inbox API
function startPancakeInboxPolling(pageId) {
    // Clear any existing interval
    if (pancakeCheckInboxInterval) {
        clearInterval(pancakeCheckInboxInterval);
        pancakeCheckInboxInterval = null;
    }
    
    if (!pageId) {
        console.log('[Extension] Không thể bắt đầu polling vì thiếu page_id');
        return;
    }
    
    // Get JWT token
    const jwtToken = getJWTFromCookies();
    if (!jwtToken) {
        console.log('[Extension] No JWT token found in cookies for Pancake polling');
        return;
    }
    
    console.log('[Extension] Starting Pancake inbox polling for page ID:', pageId);
    
    // Function to make the API call
    const checkInbox = async () => {
        try {
            const response = await fetch(`https://pancake.vn/api/v1/pages/${pageId}/check_inbox?access_token=${jwtToken}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                }
            });
            
            if (response.ok) {
                const data = await response.json();
                console.log('[Extension] Pancake check_inbox response:', data);
            } else {
                console.error('[Extension] Pancake check_inbox failed:', response.status);
            }
        } catch (error) {
            console.error('[Extension] Error during Pancake check_inbox:', error);
        }
    };
    
    // Make an initial call
    checkInbox();
    
    // Set up interval (every 3 seconds)
    pancakeCheckInboxInterval = setInterval(checkInbox, 3000);
}

// Function to stop polling
function stopPancakeInboxPolling() {
    if (pancakeCheckInboxInterval) {
        console.log('[Extension] Stopping Pancake inbox polling');
        clearInterval(pancakeCheckInboxInterval);
        pancakeCheckInboxInterval = null;
    }
}

// Polling để kiểm tra thay đổi hội thoại trên Pancake
let pancakeConversationCheckInterval = null;

function startPancakeConversationChecking() {
    if (!isPancakePage()) return;
    
    // Clear any existing interval
    if (pancakeConversationCheckInterval) {
        clearInterval(pancakeConversationCheckInterval);
        pancakeConversationCheckInterval = null;
    }
    
    console.log('[Extension] Bắt đầu polling kiểm tra thay đổi hội thoại Pancake');
    
    pancakeConversationCheckInterval = setInterval(() => {
        const currentPancakeIds = extractPancakeIds();
        
        if (pancakeIdsChanged(currentPancakeIds, lastPancakeIds)) {
            console.log('[Extension] 📊 Polling phát hiện thay đổi hội thoại:', {
                old: lastPancakeIds,
                new: currentPancakeIds
            });
            
            lastPancakeIds = currentPancakeIds;
            
            // Stop any existing polling
            stopPancakeInboxPolling();
            
            // Kiểm tra lại trạng thái AI cho hội thoại mới
            if (currentPancakeIds && currentPancakeIds.page_id && currentPancakeIds.pid) {
                const aiBtn = document.getElementById('ai-status-btn');
                if (aiBtn) {
                    // Nếu nút đã tồn tại, chỉ cần kiểm tra lại trạng thái
                    const nameElement = document.querySelector('.copyable-text.customer-name');
                    const fullname = nameElement ? nameElement.textContent.trim() : '';
                    
                    console.log('[Extension] 🔍 Kiểm tra trạng thái AI cho hội thoại mới qua polling');
                    checkAIStatus(aiBtn, currentPancakeIds.page_id, currentPancakeIds.pid, fullname);
                    
                    // Khởi động lại polling
                    startPancakeInboxPolling(currentPancakeIds.page_id);
                } else {
                    // Nếu nút chưa tồn tại, khởi tạo lại
                    safeInitialize();
                }
            }
        }
    }, 500); // Giảm từ 1000ms xuống 500ms để phản hồi nhanh hơn
}

function stopPancakeConversationChecking() {
    if (pancakeConversationCheckInterval) {
        console.log('[Extension] Dừng polling kiểm tra thay đổi hội thoại Pancake');
        clearInterval(pancakeConversationCheckInterval);
        pancakeConversationCheckInterval = null;
    }
}

// Lắng nghe tin nhắn từ popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.type === 'check_ai_button') {
        checkAndReinitializeAIButton();
        sendResponse({ success: true });
    }
});

// Theo dõi thay đổi URL và hội thoại
let lastURL = location.href;
let lastPancakeIds = null; // Lưu trạng thái hội thoại trước đó

// Hàm so sánh pancakeIds để phát hiện thay đổi hội thoại
function pancakeIdsChanged(newIds, oldIds) {
    if (!newIds && !oldIds) return false;
    if (!newIds || !oldIds) return true;
    return newIds.page_id !== oldIds.page_id || newIds.pid !== oldIds.pid;
}

new MutationObserver(() => {
    const currentURL = location.href;
    let shouldReinitialize = false;
    
    // Kiểm tra thay đổi URL
    if (currentURL !== lastURL) {
        lastURL = currentURL;
        console.log('[Extension] URL đã thay đổi, khởi tạo lại...');
        shouldReinitialize = true;
    }
    
    // Kiểm tra thay đổi hội thoại trên Pancake
    if (isPancakePage()) {
        const currentPancakeIds = extractPancakeIds();
        if (pancakeIdsChanged(currentPancakeIds, lastPancakeIds)) {
            console.log('[Extension] Hội thoại Pancake đã thay đổi:', {
                old: lastPancakeIds,
                new: currentPancakeIds
            });
            lastPancakeIds = currentPancakeIds;
            shouldReinitialize = true;
        }
    }
    
    if (shouldReinitialize) {
        // Stop any existing polling
        stopPancakeInboxPolling();
        stopPancakeConversationChecking();
        
        // Đợi một chút để DOM cập nhật, sau đó xóa tất cả nút hiện có và khởi tạo lại
        setTimeout(() => {
            // Xóa tất cả nút hiện có
            document.querySelectorAll('#ai-status-btn').forEach(btn => btn.remove());
            safeInitialize();
            
            // Khởi động conversation checking nếu ở Pancake
            if (isPancakePage()) {
                startPancakeConversationChecking();
            }
        }, 500);
    }
}).observe(document, { subtree: true, childList: true });

// Lắng nghe sự kiện window load để đảm bảo khởi tạo khi tải lại trang
window.addEventListener('load', () => {
    console.log('[Extension] Trang đã tải hoàn tất, khởi tạo lại...');
    // Xóa tất cả nút hiện có
    document.querySelectorAll('#ai-status-btn').forEach(btn => btn.remove());
    
    // Stop any existing polling
    stopPancakeInboxPolling();
    stopPancakeConversationChecking();
    
    // Đợi DOM cập nhật
    setTimeout(() => {
        safeInitialize();
        // Khởi tạo chức năng kiểm tra liên tục
        initContinuousButtonCheck();
        
        // Khởi động conversation checking cho Pancake
        if (isPancakePage()) {
            startPancakeConversationChecking();
        }
    }, 1000);
});

// Khởi tạo AI button - sửa để chạy cả Facebook và Pancake
setTimeout(() => {
    // Xóa tất cả nút hiện có
    document.querySelectorAll('#ai-status-btn').forEach(btn => btn.remove());
    safeInitialize();
    // Khởi tạo chức năng kiểm tra liên tục
    initContinuousButtonCheck();
    // Khởi tạo theo dõi avatar cho Pancake
    initAvatarObserver();
    // Khởi tạo theo dõi danh sách chat cho Pancake
    initChatListObserver();
    // Khởi tạo theo dõi selected-customer cho Pancake
    initSelectedCustomerObserver();
    // Khởi động conversation checking cho Pancake
    if (isPancakePage()) {
        startPancakeConversationChecking();
    }
}, 500);

// Theo dõi thay đổi avatar trên Pancake
function initAvatarObserver() {
    if (!isPancakePage()) return;
    
    console.log('[Extension] Khởi tạo theo dõi avatar trên Pancake');
    
    let avatarChangeTimeout = null;
    
    const handleAvatarChange = () => {
        // Debounce để tránh gọi quá nhiều lần
        if (avatarChangeTimeout) {
            clearTimeout(avatarChangeTimeout);
        }
        
        avatarChangeTimeout = setTimeout(() => {
            console.log('[Extension] 🔄 Avatar đã thay đổi, kiểm tra hội thoại mới');
            
            // Kiểm tra xem có phải hội thoại mới không
            const currentPancakeIds = extractPancakeIds();
            if (pancakeIdsChanged(currentPancakeIds, lastPancakeIds)) {
                console.log('[Extension] 🆕 Phát hiện hội thoại mới từ avatar change:', {
                    old: lastPancakeIds,
                    new: currentPancakeIds
                });
                lastPancakeIds = currentPancakeIds;
                
                // Stop any existing polling
                stopPancakeInboxPolling();
                
                // Xóa và tạo lại nút AI
                document.querySelectorAll('#ai-status-btn').forEach(btn => btn.remove());
                safeInitialize();
            }
        }, 100);
    };
    
    // Observer 1: Theo dõi thay đổi src của avatar images
    const avatarObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
            if (mutation.type === 'attributes' && mutation.attributeName === 'src' && 
                mutation.target.tagName === 'IMG' && 
                (mutation.target.getAttribute('src')?.includes('/avatar') || 
                 mutation.target.className?.includes('customer-avatar'))) {
                console.log('[Extension] 🖼️ Avatar src changed:', mutation.target.src);
                handleAvatarChange();
                break;
            }
        }
    });
    
    // Observer 2: Theo dõi thay đổi class của avatar images
    const classObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
            if (mutation.type === 'attributes' && mutation.attributeName === 'class' && 
                mutation.target.tagName === 'IMG' && 
                mutation.target.className?.includes('customer-avatar')) {
                console.log('[Extension] 🎨 Avatar class changed:', mutation.target.className);
                handleAvatarChange();
                break;
            }
        }
    });
    
    // Observer 3: Theo dõi thay đổi nội dung của selected-customer
    const selectedCustomerObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
            if (mutation.type === 'childList') {
                // Kiểm tra nếu có avatar mới được thêm vào
                mutation.addedNodes.forEach(node => {
                    if (node.nodeType === Node.ELEMENT_NODE) {
                        const avatarInNode = node.querySelector ? node.querySelector('img[class*="customer-avatar"]') : null;
                        if (avatarInNode || (node.tagName === 'IMG' && node.className?.includes('customer-avatar'))) {
                            console.log('[Extension] 🆕 New avatar added to DOM');
                            handleAvatarChange();
                        }
                    }
                });
            }
        }
    });
    
    // Theo dõi tất cả img avatar hiện có
    document.querySelectorAll('img[src*="/avatar"], img[class*="customer-avatar"]').forEach(img => {
        avatarObserver.observe(img, { attributes: true, attributeFilter: ['src'] });
        classObserver.observe(img, { attributes: true, attributeFilter: ['class'] });
    });
    
    // Theo dõi selected-customer container
    const selectedCustomer = document.querySelector('.selected-customer');
    if (selectedCustomer) {
        selectedCustomerObserver.observe(selectedCustomer, { 
            childList: true, 
            subtree: true 
        });
        console.log('[Extension] 👀 Đang theo dõi .selected-customer');
    }
    
    // Theo dõi chat-menu container để bắt các avatar mới
    const chatMenu = document.querySelector('.chat-menu');
    if (chatMenu) {
        selectedCustomerObserver.observe(chatMenu, { 
            childList: true, 
            subtree: true 
        });
        console.log('[Extension] 👀 Đang theo dõi .chat-menu');
    }
    
    // Observer cho thay đổi customer name
    const nameObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
            if (mutation.type === 'characterData' || 
                (mutation.type === 'childList' && mutation.target.classList?.contains('customer-name'))) {
                console.log('[Extension] 📝 Customer name changed');
                handleAvatarChange();
                break;
            }
        }
    });
    
    // Theo dõi thay đổi tên khách hàng
    const customerNameElements = document.querySelectorAll('.customer-name, [class*="customer-name"]');
    customerNameElements.forEach(element => {
        nameObserver.observe(element, { 
            attributes: true, 
            childList: true, 
            subtree: true,
            characterData: true 
        });
    });
    
    // Theo dõi thay đổi DOM toàn trang để bắt các element mới
    const globalObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
            if (mutation.type === 'childList') {
                mutation.addedNodes.forEach(node => {
                    if (node.nodeType === Node.ELEMENT_NODE && node.querySelectorAll) {
                        // Thêm observer cho các avatar mới
                        node.querySelectorAll('img[src*="/avatar"], img[class*="customer-avatar"]').forEach(img => {
                            avatarObserver.observe(img, { attributes: true, attributeFilter: ['src'] });
                            classObserver.observe(img, { attributes: true, attributeFilter: ['class'] });
                        });
                        
                        // Thêm observer cho customer name mới
                        node.querySelectorAll('.customer-name, [class*="customer-name"]').forEach(element => {
                            nameObserver.observe(element, { 
                                attributes: true, 
                                childList: true, 
                                subtree: true,
                                characterData: true 
                            });
                        });
                    }
                });
            }
        }
    });
    
    globalObserver.observe(document.body, { childList: true, subtree: true });
}

// Function to extract page_id from URL for Pancake
function extractPageIdFromUrl() {
    if (!isPancakePage()) return null;
    
    // Try to extract from URL path first
    const pathMatch = window.location.pathname.match(/\/pages\/(\d+)/);
    if (pathMatch && pathMatch[1]) {
        return pathMatch[1];
    }
    
    // If not in path, try from query parameters
    const searchParams = new URLSearchParams(window.location.search);
    const pageIdFromQuery = searchParams.get('page_id');
    if (pageIdFromQuery) {
        return pageIdFromQuery;
    }
    
    return null;
}

// Function to start Pancake inbox polling as early as possible
function initEarlyPancakePolling() {
    if (!isPancakePage()) return;
    
    console.log('[Extension] Attempting early Pancake polling initialization');
    
    // Try to get page_id from URL first
    let pageId = extractPageIdFromUrl();
    
    // If not found in URL, try to extract from DOM elements
    if (!pageId) {
        const pancakeIds = extractPancakeIds();
        if (pancakeIds && pancakeIds.page_id) {
            pageId = pancakeIds.page_id;
        }
    }
    
    // If we have a page_id, start polling
    if (pageId) {
        console.log('[Extension] Early detection of Pancake page_id:', pageId);
        startPancakeInboxPolling(pageId);
        return true;
    }
    
    console.log('[Extension] Couldn\'t detect Pancake page_id early, will try again later');
    return false;
}

// Thêm một early check khi DOM sẵn sàng
function earlyInitCheck() {
    console.log('[Extension] Thực hiện kiểm tra sớm khi DOM sẵn sàng');
    
    // Thử kích hoạt polling Pancake sớm nhất có thể
    const pollingStarted = initEarlyPancakePolling();
    
    // Khởi tạo các observer cho Pancake ngay từ đầu
    if (isPancakePage()) {
        initAvatarObserver();
        initChatListObserver();
        initSelectedCustomerObserver();
        // Bắt đầu polling kiểm tra thay đổi hội thoại
        startPancakeConversationChecking();
    }
    
    // Nếu không thể bắt đầu polling ngay, thiết lập một interval để thử lại
    if (!pollingStarted && isPancakePage()) {
        let attempts = 0;
        const earlyPollInterval = setInterval(() => {
            attempts++;
            if (attempts > 10) {
                clearInterval(earlyPollInterval);
                console.log('[Extension] Không thể bắt đầu polling sớm sau nhiều lần thử');
                return;
            }
            
            if (initEarlyPancakePolling()) {
                clearInterval(earlyPollInterval);
                console.log('[Extension] Đã bắt đầu polling sau', attempts, 'lần thử');
            }
        }, 1000);
    }
}

// Gọi earlyInitCheck ngay khi script được thực thi
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', earlyInitCheck);
} else {
    earlyInitCheck();
}

// Theo dõi thay đổi trong danh sách chat trên Pancake
function initChatListObserver() {
    if (!isPancakePage()) return;
    
    console.log('[Extension] Khởi tạo theo dõi danh sách chat trên Pancake');
    
    let chatChangeTimeout = null;
    
    const handleChatListChange = () => {
        // Debounce để tránh gọi quá nhiều lần
        if (chatChangeTimeout) {
            clearTimeout(chatChangeTimeout);
        }
        
        chatChangeTimeout = setTimeout(() => {
            const currentPancakeIds = extractPancakeIds();
            if (pancakeIdsChanged(currentPancakeIds, lastPancakeIds)) {
                console.log('[Extension] Phát hiện chuyển hội thoại từ chat list:', {
                    old: lastPancakeIds,
                    new: currentPancakeIds
                });
                lastPancakeIds = currentPancakeIds;
                
                // Stop any existing polling
                stopPancakeInboxPolling();
                
                // Xóa và tạo lại nút AI
                document.querySelectorAll('#ai-status-btn').forEach(btn => btn.remove());
                safeInitialize();
            }
        }, 100); // Giảm thời gian debounce để phản hồi nhanh hơn
    };
    
    // Theo dõi click events trên các chat items
    const handleChatItemClick = (event) => {
        // Kiểm tra xem có phải click vào chat item không
        const chatItem = event.target.closest('[class*="chat"], [class*="conversation"], [class*="item"]');
        if (chatItem) {
            console.log('[Extension] Phát hiện click vào chat item');
            setTimeout(handleChatListChange, 50); // Đợi DOM cập nhật sau khi click
        }
    };
    
    // Thêm event listener cho click
    document.addEventListener('click', handleChatItemClick, true);
    
    // Theo dõi thay đổi trong sidebar hoặc chat list
    const chatListObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
            // Kiểm tra nếu có thay đổi trong class 'active' hoặc 'selected'
            if (mutation.type === 'attributes' && 
                (mutation.attributeName === 'class' || mutation.attributeName === 'data-selected')) {
                const target = mutation.target;
                if (target.classList?.contains('active') || 
                    target.classList?.contains('selected') ||
                    target.getAttribute('data-selected') === 'true') {
                    console.log('[Extension] Phát hiện thay đổi class active/selected');
                    handleChatListChange();
                    break;
                }
            }
            
            // Kiểm tra nếu có element mới được thêm vào với class active/selected
            if (mutation.type === 'childList') {
                mutation.addedNodes.forEach(node => {
                    if (node.classList && 
                        (node.classList.contains('active') || node.classList.contains('selected'))) {
                        console.log('[Extension] Phát hiện element mới với class active/selected');
                        handleChatListChange();
                    }
                });
            }
        }
    });
    
    // Tìm và theo dõi sidebar hoặc chat list container
    const chatListContainers = document.querySelectorAll('[class*="chat"], [class*="conversation"], [class*="sidebar"]');
    chatListContainers.forEach(container => {
        chatListObserver.observe(container, {
            attributes: true,
            childList: true,
            subtree: true,
            attributeFilter: ['class', 'data-selected', 'aria-selected']
        });
    });
    
    // Cũng theo dõi toàn bộ body để bắt các thay đổi động
    chatListObserver.observe(document.body, {
        attributes: true,
        childList: true,
        subtree: true,
        attributeFilter: ['class', 'data-selected', 'aria-selected']
    });
    
    // Theo dõi thay đổi URL hash nếu Pancake sử dụng hash routing
    let lastHash = location.hash;
    setInterval(() => {
        if (location.hash !== lastHash) {
            lastHash = location.hash;
            console.log('[Extension] Hash thay đổi, kiểm tra hội thoại mới');
            handleChatListChange();
        }
    }, 500);
}

// Observer đặc biệt để theo dõi selected-customer
function initSelectedCustomerObserver() {
    if (!isPancakePage()) return;
    
    console.log('[Extension] 🎯 Khởi tạo observer cho selected-customer');
    
    let changeTimeout = null;
    
    const handleSelectedCustomerChange = () => {
        if (changeTimeout) {
            clearTimeout(changeTimeout);
        }
        
        changeTimeout = setTimeout(() => {
            console.log('[Extension] 👤 Selected customer changed');
            const currentPancakeIds = extractPancakeIds();
            
            if (pancakeIdsChanged(currentPancakeIds, lastPancakeIds)) {
                console.log('[Extension] ✨ Phát hiện thay đổi hội thoại từ selected-customer:', {
                    old: lastPancakeIds,
                    new: currentPancakeIds
                });
                
                lastPancakeIds = currentPancakeIds;
                
                // Stop any existing polling
                stopPancakeInboxPolling();
                
                // Kiểm tra lại trạng thái AI ngay lập tức
                const aiBtn = document.getElementById('ai-status-btn');
                if (aiBtn && currentPancakeIds && currentPancakeIds.page_id && currentPancakeIds.pid) {
                    const nameElement = document.querySelector('.copyable-text.customer-name');
                    const fullname = nameElement ? nameElement.textContent.trim() : '';
                    
                    console.log('[Extension] 🔍 Kiểm tra trạng thái AI cho hội thoại mới từ selected-customer');
                    checkAIStatus(aiBtn, currentPancakeIds.page_id, currentPancakeIds.pid, fullname);
                    
                    // Khởi động lại polling
                    startPancakeInboxPolling(currentPancakeIds.page_id);
                } else {
                    // Nếu nút chưa tồn tại, khởi tạo lại
                    document.querySelectorAll('#ai-status-btn').forEach(btn => btn.remove());
                    safeInitialize();
                }
            }
        }, 50); // Thời gian debounce rất ngắn để phản hồi nhanh
    };
    
    // Observer cho selected-customer container
    const selectedCustomerObserver = new MutationObserver((mutations) => {
        let hasRelevantChange = false;
        
        for (const mutation of mutations) {
            // Kiểm tra thay đổi attributes
            if (mutation.type === 'attributes') {
                if (mutation.target.tagName === 'IMG' && 
                    (mutation.attributeName === 'src' || mutation.attributeName === 'class') &&
                    (mutation.target.getAttribute('src')?.includes('/avatar') || 
                     mutation.target.className?.includes('customer-avatar'))) {
                    console.log('[Extension] 🖼️ Avatar attribute changed in selected-customer');
                    hasRelevantChange = true;
                    break;
                }
            }
            
            // Kiểm tra thay đổi childList
            if (mutation.type === 'childList') {
                // Kiểm tra nếu có node mới chứa avatar hoặc customer info
                mutation.addedNodes.forEach(node => {
                    if (node.nodeType === Node.ELEMENT_NODE) {
                        if (node.tagName === 'IMG' && 
                            (node.getAttribute('src')?.includes('/avatar') || 
                             node.className?.includes('customer-avatar'))) {
                            console.log('[Extension] 🆕 New avatar added to selected-customer');
                            hasRelevantChange = true;
                        } else if (node.querySelector && 
                                  (node.querySelector('img[src*="/avatar"]') || 
                                   node.querySelector('img[class*="customer-avatar"]') ||
                                   node.querySelector('.customer-name'))) {
                            console.log('[Extension] 🆕 New customer info added to selected-customer');
                            hasRelevantChange = true;
                        }
                    }
                });
            }
            
            // Kiểm tra thay đổi text content của customer name
            if (mutation.type === 'characterData' && 
                mutation.target.parentElement?.classList?.contains('customer-name')) {
                console.log('[Extension] 📝 Customer name text changed');
                hasRelevantChange = true;
            }
        }
        
        if (hasRelevantChange) {
            handleSelectedCustomerChange();
        }
    });
    
    // Tìm và observe selected-customer
    const selectedCustomer = document.querySelector('.selected-customer');
    if (selectedCustomer) {
        selectedCustomerObserver.observe(selectedCustomer, {
            attributes: true,
            childList: true,
            subtree: true,
            characterData: true,
            attributeFilter: ['src', 'class']
        });
        console.log('[Extension] ✅ Đã thiết lập observer cho .selected-customer');
    } else {
        console.log('[Extension] ⚠️ Không tìm thấy .selected-customer');
    }
    
    // Observer để theo dõi khi selected-customer được tạo mới
    const bodyObserver = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
            if (mutation.type === 'childList') {
                mutation.addedNodes.forEach(node => {
                    if (node.nodeType === Node.ELEMENT_NODE && node.querySelectorAll) {
                        const newSelectedCustomer = node.querySelector('.selected-customer');
                        if (newSelectedCustomer) {
                            console.log('[Extension] 🆕 New selected-customer found, setting up observer');
                            selectedCustomerObserver.observe(newSelectedCustomer, {
                                attributes: true,
                                childList: true,
                                subtree: true,
                                characterData: true,
                                attributeFilter: ['src', 'class']
                            });
                            handleSelectedCustomerChange();
                        }
                    }
                });
            }
        }
    });
    
    bodyObserver.observe(document.body, { childList: true, subtree: true });
}

// Khởi tạo tất cả các observer cần thiết cho Pancake
function initAllObservers() {
    if (!isPancakePage()) return;
    
    console.log('[Extension] Khởi tạo tất cả các observer cho Pancake');
    
    // Khởi tạo observer cho avatar
    initAvatarObserver();
    
    // Khởi tạo observer cho danh sách chat
    initChatListObserver();
    
    // Khởi tạo observer cho selected-customer
    initSelectedCustomerObserver();
}

// Gọi hàm khởi tạo observer khi trang được tải
window.addEventListener('load', () => {
    setTimeout(() => {
        initAllObservers();
    }, 1000);
});